/*
DayPilot Lite
Copyright (c) 2005 - 2026 Annpoint s.r.o.
https://www.daypilot.org/
Licensed under Apache Software License 2.0
Version: 2026.1.796-lite
*/
"use strict";
(function (DayPilot) {
    'use strict';
    if (typeof DayPilot.Month !== 'undefined' && DayPilot.Month.events) {
        return;
    }
    var DayPilotMonth = {};
    var isVueVNode = DayPilot.Util.isVueVNode;
    DayPilotMonth.Month = function (placeholder, options) {
        this.v = '${v}';
        this.nav = {};
        var calendar = this;
        this.id = placeholder;
        this.isMonth = true;
        this.api = 2;
        this.backendUrl = null;
        this.cellHeaderHeight = 24;
        this.cellHeight = 100;
        this.cellMarginBottom = 0;
        this.contextMenu = null;
        this.cssClassPrefix = "month_default";
        this.eventBarVisible = true;
        this.eventBorderRadius = null;
        this.eventHeight = 25;
        this.eventsLoadMethod = "GET";
        this.headerHeight = 30;
        this.hideUntilInit = true;
        this.lineSpace = 1;
        this.locale = "en-us";
        this.showToolTip = true;
        this.startDate = new DayPilot.Date();
        this.theme = null;
        this.visible = true;
        this.weekStarts = "Auto";
        this.width = '100%';
        this.xssProtection = "Enabled";
        this.afterRender = function () { };
        this.cellHeaderClickHandling = "Enabled";
        this.eventClickHandling = "Enabled";
        this.eventDeleteHandling = "Disabled";
        this.eventMoveHandling = "Update";
        this.eventResizeHandling = "Update";
        this.eventRightClickHandling = "ContextMenu";
        this.headerClickHandling = "Enabled";
        this.timeRangeSelectedHandling = "Enabled";
        this.onCellHeaderClick = null;
        this.onCellHeaderClicked = null;
        this.onEventClick = null;
        this.onEventClicked = null;
        this.onEventDelete = null;
        this.onEventDeleted = null;
        this.onEventMove = null;
        this.onEventMoved = null;
        this.onEventResize = null;
        this.onEventResized = null;
        this.onEventRightClick = null;
        this.onEventRightClicked = null;
        this.onTimeRangeSelect = null;
        this.onTimeRangeSelected = null;
        this.onBeforeEventRender = null;
        this.onBeforeCellRender = null;
        this.cellEvents = [];
        this.elements = {};
        this.elements.events = [];
        this.cache = {};
        this._disposed = false;
        this._resolved = {};
        var resolved = this._resolved;
        this._updateView = function (json) {
            var result = JSON.parse(json);
            if (result.CallBackRedirect) {
                document.location.href = result.CallBackRedirect;
                return;
            }
            if (result.UpdateType === "None") {
                calendar.fireAfterRenderDetached(result.CallBackData, true);
                return;
            }
            calendar.events.list = result.Events;
            if (result.UpdateType === "Full") {
                calendar.startDate = result.StartDate;
                calendar.timeFormat = result.TimeFormat ? result.TimeFormat : calendar.timeFormat;
                if (typeof result.WeekStarts !== 'undefined') {
                    calendar.weekStarts = result.WeekStarts;
                }
                calendar.hashes = result.Hashes;
            }
            calendar._deleteEvents();
            calendar._prepareRows();
            calendar._loadEvents();
            if (result.UpdateType === "Full") {
                calendar._clearTable();
                calendar._drawTable();
            }
            calendar._updateHeight();
            calendar.show();
            calendar._drawEvents();
            calendar.fireAfterRenderDetached(result.CallBackData, true);
        };
        this.fireAfterRenderDetached = function (data, isCallBack) {
            var afterRenderDelayed = function (data, isc) {
                return function () {
                    if (calendar.afterRender) {
                        calendar.afterRender(data, isc);
                    }
                };
            };
            window.setTimeout(afterRenderDelayed(data, isCallBack), 0);
        };
        this.lineHeight = function () {
            return this.eventHeight + this.lineSpace;
        };
        this.events = {};
        this.events.add = function (e) {
            var data = null;
            if (e instanceof DayPilot.Event) {
                data = e.data;
            }
            else if (typeof e === "object") {
                data = e;
            }
            else {
                throw "DayPilot.Month.events.add() expects an object or DayPilot.Event instance.";
            }
            if (!calendar.events.list) {
                calendar.events.list = [];
            }
            calendar.events.list.push(data);
            calendar.update();
            calendar._angular.notify();
        };
        this.events.find = function (id) {
            if (!calendar.events.list) {
                return null;
            }
            if (typeof id === "function") {
                var fn = id;
                for (var i = 0; i < calendar.events.list.length; i++) {
                    var e = new DayPilot.Event(calendar.events.list[i], calendar);
                    if (fn(e)) {
                        return e;
                    }
                }
                return null;
            }
            for (var i = 0; i < calendar.events.list.length; i++) {
                var data = calendar.events.list[i];
                if (data.id === id) {
                    return new DayPilot.Event(data, calendar);
                }
            }
            return null;
        };
        this.events.update = function (e) {
            if (e instanceof DayPilot.Event) {
                e.commit();
            }
            else if (typeof e === "object") {
                var target = calendar.events.find(e.id);
                if (target) {
                    var index = DayPilot.indexOf(calendar.events.list, target.data);
                    calendar.events.list.splice(index, 1, e);
                }
            }
            calendar.update();
            calendar._angular.notify();
        };
        this.events.remove = function (e) {
            var data;
            if (e instanceof DayPilot.Event) {
                data = e.data;
            }
            else if (typeof e === "object") {
                var target = calendar.events.find(e.id);
                if (target) {
                    data = target.data;
                }
            }
            else if (typeof e === "string" || typeof e === "number") {
                var target = calendar.events.find(e);
                if (target) {
                    data = target.data;
                }
            }
            var index = DayPilot.indexOf(calendar.events.list, data);
            calendar.events.list.splice(index, 1);
            calendar.update();
            calendar._angular.notify();
        };
        this.events.load = function (url, success, error) {
            var onError = function (args) {
                var largs = {};
                largs.exception = args.exception;
                largs.request = args.request;
                if (typeof error === 'function') {
                    error(largs);
                }
            };
            var onSuccess = function (args) {
                var r = args.request;
                var data;
                try {
                    data = JSON.parse(r.responseText);
                }
                catch (e) {
                    var fargs = {};
                    fargs.exception = e;
                    onError(fargs);
                    return;
                }
                if (DayPilot.isArray(data)) {
                    var sargs = {};
                    sargs.preventDefault = function () {
                        this.preventDefault.value = true;
                    };
                    sargs.data = data;
                    if (typeof success === "function") {
                        success(sargs);
                    }
                    if (sargs.preventDefault.value) {
                        return;
                    }
                    calendar.events.list = data;
                    if (calendar._initialized) {
                        calendar.update();
                    }
                }
            };
            var usePost = calendar.eventsLoadMethod && calendar.eventsLoadMethod.toUpperCase() === "POST";
            if (usePost) {
                DayPilot.Http.ajax({
                    "method": "POST",
                    "data": { "start": calendar.visibleStart().toString(), "end": calendar.visibleEnd().toString() },
                    "url": url,
                    "success": onSuccess,
                    "error": onError
                });
            }
            else {
                var fullUrl = url;
                var queryString = "start=" + calendar.visibleStart().toString() + "&end=" + calendar.visibleEnd().toString();
                if (fullUrl.indexOf("?") > -1) {
                    fullUrl += "&" + queryString;
                }
                else {
                    fullUrl += "?" + queryString;
                }
                DayPilot.Http.ajax({
                    "method": "GET",
                    "url": fullUrl,
                    "success": onSuccess,
                    "error": onError
                });
            }
        };
        this.events.forRange = function (start, end) {
            start = new DayPilot.Date(start);
            end = new DayPilot.Date(end);
            var events = (calendar.events.list || []).map(function (data) { return new DayPilot.Event(data); });
            events.sort(calendar._eventComparer);
            return events.filter(function (e) {
                var estart = e.start();
                var eend = e.end();
                var startPointOnly = estart === eend && estart === start;
                return startPointOnly || DayPilot.Util.overlaps(start, end, estart, eend);
            });
        };
        this.update = function (options) {
            calendar._loadOptions(options);
            if (!this._initialized) {
                return;
            }
            if (calendar._disposed) {
                throw new DayPilot.Exception("You are trying to update a DayPilot.Month instance that has been disposed.");
            }
            if (!this.cells) {
                return;
            }
            var full = true;
            calendar._deleteEvents();
            calendar._prepareRows();
            calendar._loadEvents();
            if (full) {
                calendar._clearTable();
                calendar._drawTable();
            }
            calendar._updateHeight();
            calendar._show();
            calendar._drawEvents();
            if (this.visible) {
                this.show();
            }
            else {
                this.hide();
            }
        };
        this._specialHandling = null;
        this._loadOptions = function (options) {
            if (!options) {
                return;
            }
            var specialHandling = {
                "events": {
                    "preInit": function () {
                        var events = this.data || [];
                        if (DayPilot.isArray(events.list)) {
                            calendar.events.list = events.list;
                        }
                        else {
                            calendar.events.list = events;
                        }
                    }
                }
            };
            this._specialHandling = specialHandling;
            for (var name_1 in options) {
                if (specialHandling[name_1]) {
                    var item = specialHandling[name_1];
                    item.data = options[name_1];
                    if (item.preInit) {
                        item.preInit();
                    }
                }
                else {
                    calendar[name_1] = options[name_1];
                }
            }
        };
        this._postInit = function () {
            var specialHandling = this._specialHandling;
            for (var name_2 in specialHandling) {
                var item = specialHandling[name_2];
                if (item.postInit) {
                    item.postInit();
                }
            }
        };
        this._cache = {};
        this._cache.events = [];
        this._doBeforeEventRender = function (i) {
            var cache = this._cache.events;
            var data = this.events.list[i];
            var evc = {};
            for (var name_3 in data) {
                evc[name_3] = data[name_3];
            }
            if (typeof this.onBeforeEventRender === 'function') {
                var args = {};
                args.control = calendar;
                args.data = evc;
                this.onBeforeEventRender(args);
            }
            cache[i] = evc;
        };
        this._loadEvents = function () {
            var events = this.events.list;
            if (!events) {
                return;
            }
            if (!DayPilot.isArray(events)) {
                throw new DayPilot.Exception("DayPilot.Month.events.list expects an array object. You supplied: " + (typeof events));
            }
            if (typeof this.onBeforeEventRender === 'function') {
                for (var i = 0; i < events.length; i++) {
                    this._doBeforeEventRender(i);
                }
            }
            for (var x = 0; x < events.length; x++) {
                var data = events[x];
                if (typeof data !== "object") {
                    throw new DayPilot.Exception("Event data item must be an object");
                }
                if (!data.start) {
                    throw new DayPilot.Exception("Event data item must specify 'start' property");
                }
                if (!data.end) {
                    throw new DayPilot.Exception("Event data item must specify 'end' property");
                }
                var start = new DayPilot.Date(data.start);
                var end = new DayPilot.Date(data.end);
                if (start.getTime() > end.getTime()) {
                    continue;
                }
                for (var i = 0; i < this.rows.length; i++) {
                    var row = this.rows[i];
                    var ep = new DayPilot.Event(data, this);
                    if (row.belongsHere(ep)) {
                        row.events.push(ep);
                        if (typeof this.onBeforeEventRender === 'function') {
                            ep.cache = this._cache.events[x];
                        }
                    }
                }
            }
            for (var ri = 0; ri < this.rows.length; ri++) {
                var row = this.rows[ri];
                row.events.sort(this._eventComparer);
                for (var ei = 0; ei < this.rows[ri].events.length; ei++) {
                    var ev = row.events[ei];
                    var colStart = row.getStartColumn(ev);
                    var colWidth = row.getWidth(ev);
                    row.putIntoLine(ev, colStart, colWidth, ri);
                }
            }
        };
        this._deleteEvents = function () {
            for (var i = 0; i < this.elements.events.length; i++) {
                var e = this.elements.events[i];
                calendar._deleteEvent(e);
            }
            this.elements.events = [];
        };
        this._deleteEvent = function (div) {
            (function domRemove() {
                var domArgs = div.domArgs;
                div.domArgs = null;
                if (domArgs && typeof calendar.onBeforeEventDomRemove === "function") {
                    calendar.onBeforeEventDomRemove(domArgs);
                }
                if (domArgs && typeof calendar.onBeforeEventDomAdd === "function") {
                    var target = domArgs && domArgs._targetElement;
                    if (target) {
                        var isVue = calendar._vue._vueImport && isVueVNode(domArgs.element);
                        if (isVue) {
                            calendar._vue._renderingEvent = true;
                            calendar._vue._unmountVueComponent(target);
                            calendar._vue._renderingEvent = false;
                        }
                    }
                }
            })();
            div.event = null;
            div.click = null;
            div.parentNode.removeChild(div);
        };
        this._drawEvents = function () {
            this._drawEventsRows();
        };
        this._drawEventsRows = function () {
            this.elements.events = [];
            for (var ri = 0; ri < this.rows.length; ri++) {
                var row = this.rows[ri];
                for (var li = 0; li < row.lines.length; li++) {
                    var line = row.lines[li];
                    for (var pi = 0; pi < line.length; pi++) {
                        this._drawEvent(line[pi]);
                    }
                }
            }
        };
        this._eventComparer = function (a, b) {
            if (!a || !b || !a.start || !b.start) {
                return 0;
            }
            var byStart = a.start().getTime() - b.start().getTime();
            if (byStart !== 0) {
                return byStart;
            }
            var byEnd = b.end().getTime() - a.end().getTime();
            return byEnd;
        };
        this.drawShadow = function (x, y, line, width, offset) {
            if (!offset) {
                offset = 0;
            }
            var remains = width;
            this.shadow = {};
            this.shadow.list = [];
            this.shadow.start = { x: x, y: y };
            this.shadow.width = width;
            var hidden = y * 7 + x - offset;
            if (hidden < 0) {
                remains += hidden;
                x = 0;
                y = 0;
            }
            var remainingOffset = offset;
            while (remainingOffset >= 7) {
                y--;
                remainingOffset -= 7;
            }
            if (remainingOffset > x) {
                var plus = 7 - this.getColCount();
                if (remainingOffset > (x + plus)) {
                    y--;
                    x = x + 7 - remainingOffset;
                }
                else {
                    remains = remains - remainingOffset + x;
                    x = 0;
                }
            }
            else {
                x -= remainingOffset;
            }
            if (y < 0) {
                y = 0;
                x = 0;
            }
            var cursor = null;
            if (DayPilotMonth.resizingEvent) {
                cursor = 'w-resize';
            }
            else if (DayPilotMonth.movingEvent) {
                cursor = "move";
            }
            this.nav.top.style.cursor = cursor;
            var eventBorderRadius = calendar.eventBorderRadius;
            if (typeof eventBorderRadius === "number") {
                eventBorderRadius += "px";
            }
            while (remains > 0 && y < this.rows.length) {
                var drawNow = Math.min(this.getColCount() - x, remains);
                var row = this.rows[y];
                var top_1 = this.getRowTop(y);
                var height = row.getHeight();
                var shadow = document.createElement("div");
                shadow.setAttribute("unselectable", "on");
                shadow.style.position = 'absolute';
                shadow.style.left = (this.getCellWidth() * x) + '%';
                shadow.style.width = (this.getCellWidth() * drawNow) + '%';
                shadow.style.top = (top_1) + 'px';
                shadow.style.height = (height) + 'px';
                shadow.style.cursor = cursor;
                shadow.classList.add(calendar._prefixCssClass("_shadow"));
                var inside = document.createElement("div");
                inside.setAttribute("unselectable", "on");
                shadow.appendChild(inside);
                inside.style.position = "absolute";
                inside.style.top = "0px";
                inside.style.right = "0px";
                inside.style.left = "0px";
                inside.style.bottom = "0px";
                inside.classList.add(calendar._prefixCssClass("_shadow_inner"));
                if (eventBorderRadius) {
                    if (remains === width) {
                        shadow.style.borderTopLeftRadius = eventBorderRadius;
                        shadow.style.borderBottomLeftRadius = eventBorderRadius;
                        inside.style.borderTopLeftRadius = eventBorderRadius;
                        inside.style.borderBottomLeftRadius = eventBorderRadius;
                    }
                    if (remains <= drawNow) {
                        shadow.style.borderTopRightRadius = eventBorderRadius;
                        shadow.style.borderBottomRightRadius = eventBorderRadius;
                        inside.style.borderTopRightRadius = eventBorderRadius;
                        inside.style.borderBottomRightRadius = eventBorderRadius;
                    }
                }
                this.nav.top.appendChild(shadow);
                this.shadow.list.push(shadow);
                remains -= (drawNow + 7 - this.getColCount());
                x = 0;
                y++;
            }
        };
        this.clearShadow = function () {
            if (this.shadow) {
                for (var i = 0; i < this.shadow.list.length; i++) {
                    this.nav.top.removeChild(this.shadow.list[i]);
                }
                this.shadow = null;
                this.nav.top.style.cursor = '';
            }
        };
        this.getEventTop = function (row, line) {
            var top = this.headerHeight;
            for (var i = 0; i < row; i++) {
                top += this.rows[i].getHeight();
            }
            top += this.cellHeaderHeight;
            top += line * this.lineHeight();
            return top;
        };
        this.getDateFromCell = function (x, y) {
            return this.firstDate.addDays(y * 7 + x);
        };
        this._drawEvent = function (e) {
            var data = e.cache || e.data;
            var eventBorderRadius = data.borderRadius || calendar.eventBorderRadius;
            if (typeof eventBorderRadius === "number") {
                eventBorderRadius += "px";
            }
            var row = e.part.row;
            var line = e.part.line;
            var colStart = e.part.colStart;
            var colWidth = e.part.colWidth;
            var left = this.getCellWidth() * (colStart);
            var width = this.getCellWidth() * (colWidth);
            var top = this.getEventTop(row, line);
            var div = document.createElement("div");
            div.setAttribute("unselectable", "on");
            div.style.height = this.eventHeight + 'px';
            div.style.overflow = "hidden";
            div.className = this._prefixCssClass("_event");
            if (data.cssClass) {
                DayPilot.Util.addClass(div, data.cssClass);
            }
            if (!e.part.startsHere) {
                DayPilot.Util.addClass(div, this._prefixCssClass("_event_continueleft"));
            }
            if (!e.part.endsHere) {
                DayPilot.Util.addClass(div, this._prefixCssClass("_event_continueright"));
            }
            div.event = e;
            div.style.width = width + '%';
            div.style.position = 'absolute';
            div.style.left = left + '%';
            div.style.top = top + 'px';
            if (this.showToolTip && e.client.toolTip()) {
                div.title = e.client.toolTip();
            }
            div.onclick = calendar._eventClickDispatch;
            div.oncontextmenu = calendar._onEventContextMenu;
            div.onmousedown = function (ev) {
                ev = ev || window.event;
                var button = ev.which || ev.button;
                ev.cancelBubble = true;
                if (ev.stopPropagation) {
                    ev.stopPropagation();
                }
                if (button === 1) {
                    DayPilotMonth.movingEvent = null;
                    if (this.style.cursor === 'w-resize' || this.style.cursor === 'e-resize') {
                        var resizing = {};
                        resizing.start = {};
                        resizing.start.x = colStart;
                        resizing.start.y = row;
                        resizing.event = div.event;
                        resizing.width = DayPilot.DateUtil.daysSpan(resizing.event.start(), resizing.event.end()) + 1;
                        resizing.direction = this.style.cursor;
                        DayPilotMonth.resizingEvent = resizing;
                    }
                    else if (this.style.cursor === 'move' || e.client.moveEnabled()) {
                        calendar.clearShadow();
                        var coords = DayPilot.mo3(calendar.nav.top, ev);
                        if (!coords) {
                            return;
                        }
                        var cell = calendar.getCellBelowPoint(coords.x, coords.y);
                        var hidden = DayPilot.DateUtil.daysDiff(e.start(), calendar.rows[row].start);
                        var offset = (cell.y * 7 + cell.x) - (row * 7 + colStart);
                        if (hidden) {
                            offset += hidden;
                        }
                        var moving = {};
                        moving.start = {};
                        moving.start.x = colStart;
                        moving.start.y = row;
                        moving.start.line = line;
                        moving.offset = calendar.eventMoveToPosition ? 0 : offset;
                        moving.colWidth = colWidth;
                        moving.event = div.event;
                        moving.coords = coords;
                        DayPilotMonth.movingEvent = moving;
                    }
                }
            };
            div.onmousemove = function (ev) {
                if (typeof (DayPilotMonth) === 'undefined') {
                    return;
                }
                if (DayPilotMonth.movingEvent || DayPilotMonth.resizingEvent) {
                    return;
                }
                var offset = DayPilot.mo3(div, ev);
                if (!offset) {
                    return;
                }
                if (div.deleteIcon) {
                    div.deleteIcon.style.display = "";
                }
                var resizeMargin = 6;
                if (offset.x <= resizeMargin && e.client.resizeEnabled()) {
                    if (e.part.startsHere) {
                        div.style.cursor = "w-resize";
                        div.dpBorder = 'left';
                    }
                    else {
                        div.style.cursor = 'not-allowed';
                    }
                }
                else if (div.clientWidth - offset.x <= resizeMargin && e.client.resizeEnabled()) {
                    if (e.part.endsHere) {
                        div.style.cursor = "e-resize";
                        div.dpBorder = 'right';
                    }
                    else {
                        div.style.cursor = 'not-allowed';
                    }
                }
                else if (e.client.clickEnabled()) {
                    div.style.cursor = "pointer";
                }
                else {
                    div.style.cursor = 'default';
                }
            };
            div.onmouseleave = function () {
                if (div.deleteIcon) {
                    div.deleteIcon.style.display = "none";
                }
                div.style.cursor = '';
            };
            div.onmouseenter = function () {
                if (div.deleteIcon) {
                    div.deleteIcon.style.display = "";
                }
            };
            var inner = document.createElement("div");
            inner.setAttribute("unselectable", "on");
            inner.className = this._prefixCssClass("_event_inner");
            if (data.borderColor === "darker" && data.backColor) {
                inner.style.borderColor = DayPilot.ColorUtil.darker(data.backColor, 2);
            }
            else {
                inner.style.borderColor = data.borderColor;
            }
            if (data.backColor) {
                inner.style.background = data.backColor;
            }
            if (data.fontColor) {
                inner.style.color = data.fontColor;
            }
            if (eventBorderRadius) {
                div.style.borderRadius = eventBorderRadius;
                inner.style.borderRadius = eventBorderRadius;
            }
            div.appendChild(inner);
            if (e.client.barVisible()) {
                var bar = document.createElement("div");
                bar.setAttribute("unselectable", "on");
                bar.className = this._prefixCssClass("_event_bar");
                bar.style.position = "absolute";
                var barInner = document.createElement("div");
                barInner.setAttribute("unselectable", "on");
                barInner.className = this._prefixCssClass("_event_bar_inner");
                barInner.style.top = "0%";
                barInner.style.height = "100%";
                if (data.barColor) {
                    barInner.style.backgroundColor = data.barColor;
                }
                bar.appendChild(barInner);
                div.appendChild(bar);
            }
            if (e.client.deleteEnabled()) {
                var dheight = 18;
                var dwidth = 18;
                var dtop = Math.floor(calendar.eventHeight / 2 - dheight / 2);
                var del = document.createElement("div");
                del.style.position = "absolute";
                del.style.right = "2px";
                del.style.top = dtop + "px";
                del.style.width = dwidth + "px";
                del.style.height = dheight + "px";
                del.className = calendar._prefixCssClass("_event_delete");
                del.onmousedown = function (ev) {
                    ev.stopPropagation();
                };
                del.onclick = function (ev) {
                    ev.stopPropagation();
                    var e = this.parentNode.event;
                    if (e) {
                        calendar._eventDeleteDispatch(e);
                    }
                };
                del.style.display = "none";
                div.deleteIcon = del;
                div.appendChild(del);
            }
            var areas = data.areas ? DayPilot.Areas.copy(data.areas) : [];
            DayPilot.Areas.attach(div, e, { "areas": areas });
            if (typeof calendar.onAfterEventRender === 'function') {
                var args = {};
                args.e = div.event;
                args.div = div;
                calendar.onAfterEventRender(args);
            }
            (function domAdd() {
                var args = {};
                args.control = calendar;
                args.e = e;
                args.element = null;
                div.domArgs = args;
                if (typeof calendar.onBeforeEventDomAdd === "function") {
                    calendar.onBeforeEventDomAdd(args);
                }
                if (args.element) {
                    var target = inner;
                    if (target) {
                        args._targetElement = target;
                        var isVueNode = isVueVNode(args.element);
                        if (isVueNode) {
                            if (!calendar._vue._vueImport) {
                                throw new DayPilot.Exception("Can't reach Vue");
                            }
                            calendar._vue._renderingEvent = true;
                            calendar._vue._renderVueNode(args.element, target, { "style": { "flexGrow": 1 } });
                            calendar._vue._renderingEvent = false;
                        }
                        else {
                            target.appendChild(args.element);
                        }
                    }
                }
                else {
                    inner.innerHTML = e.client.innerHTML();
                }
            })();
            this.elements.events.push(div);
            this.nav.events.appendChild(div);
        };
        this.lastVisibleDayOfMonth = function () {
            return this.startDate.lastDayOfMonth();
        };
        this._prepareRows = function () {
            if (typeof this.startDate === 'string') {
                this.startDate = new DayPilot.Date(this.startDate);
            }
            this.startDate = this.startDate.firstDayOfMonth();
            this.firstDate = this.startDate.firstDayOfWeek(this.getWeekStart());
            var lastVisibleDayOfMonth = this.lastVisibleDayOfMonth();
            var count = DayPilot.DateUtil.daysDiff(this.firstDate, lastVisibleDayOfMonth) + 1;
            var rowCount = Math.ceil(count / 7);
            this.days = rowCount * 7;
            this.rows = [];
            var _loop_1 = function (x) {
                var r = {};
                r.start = this_1.firstDate.addDays(x * 7);
                r.end = r.start.addDays(this_1.getColCount());
                r.events = [];
                r.lines = [];
                r.index = x;
                r.minHeight = this_1.cellHeight;
                r.calendar = this_1;
                r.belongsHere = function (ev) {
                    if (ev.end().getTime() === ev.start().getTime() && ev.start().getTime() === this.start.getTime()) {
                        return true;
                    }
                    return !(ev.end().getTime() <= this.start.getTime() || ev.start().getTime() >= this.end.getTime());
                };
                r.getPartStart = function (ev) {
                    return DayPilot.DateUtil.max(this.start, ev.start());
                };
                r.getPartEnd = function (ev) {
                    return DayPilot.DateUtil.min(this.end, ev.end());
                };
                r.getStartColumn = function (ev) {
                    var partStart = this.getPartStart(ev);
                    return DayPilot.DateUtil.daysDiff(this.start, partStart);
                };
                r.getWidth = function (ev) {
                    return DayPilot.DateUtil.daysSpan(this.getPartStart(ev), this.getPartEnd(ev)) + 1;
                };
                r.putIntoLine = function (ev, colStart, colWidth, row) {
                    var thisRow = r;
                    for (var i = 0; i < this.lines.length; i++) {
                        var line_1 = this.lines[i];
                        if (line_1.isFree(colStart, colWidth)) {
                            line_1.addEvent(ev, colStart, colWidth, row, i);
                            return i;
                        }
                    }
                    var line = [];
                    line.isFree = function (colStart, colWidth) {
                        var free = true;
                        for (var i = 0; i < this.length; i++) {
                            if (!(colStart + colWidth - 1 < this[i].part.colStart || colStart > this[i].part.colStart + this[i].part.colWidth - 1)) {
                                free = false;
                            }
                        }
                        return free;
                    };
                    line.addEvent = function (ep, colStart, colWidth, row, index) {
                        ep.part.colStart = colStart;
                        ep.part.colWidth = colWidth;
                        ep.part.row = row;
                        ep.part.line = index;
                        ep.part.startsHere = thisRow.start.getTime() <= ep.start().getTime();
                        ep.part.endsHere = thisRow.end.getTime() >= ep.end().getTime();
                        this.push(ep);
                    };
                    line.addEvent(ev, colStart, colWidth, row, this.lines.length);
                    this.lines.push(line);
                    return this.lines.length - 1;
                };
                r.getHeight = function () {
                    return Math.max(this.lines.length * calendar.lineHeight() + calendar.cellHeaderHeight + calendar.cellMarginBottom, this.calendar.cellHeight);
                };
                this_1.rows.push(r);
            };
            var this_1 = this;
            for (var x = 0; x < rowCount; x++) {
                _loop_1(x);
            }
            this.endDate = this.firstDate.addDays(rowCount * 7);
        };
        this.visibleStart = function () {
            return calendar.firstDate;
        };
        this.visibleEnd = function () {
            return calendar.endDate;
        };
        this.getHeight = function () {
            var height = this.headerHeight;
            for (var i = 0; i < this.rows.length; i++) {
                height += this.rows[i].getHeight();
            }
            return height;
        };
        this.getWidth = function (start, end) {
            var diff = (end.y * 7 + end.x) - (start.y * 7 + start.x);
            return diff + 1;
        };
        this.getMinCoords = function (first, second) {
            if ((first.y * 7 + first.x) < (second.y * 7 + second.x)) {
                return first;
            }
            else {
                return second;
            }
        };
        this._prefixCssClass = function (part) {
            var prefix = this.theme || this.cssClassPrefix;
            if (prefix) {
                return prefix + part;
            }
            else {
                return "";
            }
        };
        this._drawTop = function () {
            var relative = this.nav.top;
            relative.setAttribute("unselectable", "on");
            relative.style.MozUserSelect = 'none';
            relative.style.KhtmlUserSelect = 'none';
            relative.style.WebkitUserSelect = 'none';
            relative.style.position = 'relative';
            if (this.width) {
                relative.style.width = this.width;
            }
            relative.style.height = this.getHeight() + 'px';
            relative.onselectstart = function () { return false; };
            if (this.hideUntilInit) {
                relative.style.visibility = 'hidden';
            }
            if (!this.visible) {
                relative.style.display = "none";
            }
            relative.className = this._prefixCssClass("_main");
            var cells = document.createElement("div");
            this.nav.cells = cells;
            cells.style.position = "absolute";
            cells.style.left = "0px";
            cells.style.right = "0px";
            cells.setAttribute("unselectable", "on");
            relative.appendChild(cells);
            var events = document.createElement("div");
            this.nav.events = events;
            events.style.position = "absolute";
            events.style.left = "0px";
            events.style.right = "0px";
            events.setAttribute("unselectable", "on");
            relative.appendChild(events);
            relative.onmousemove = function (ev) {
                if (DayPilotMonth.resizingEvent) {
                    var coords = DayPilot.mo3(calendar.nav.top, ev);
                    if (!coords) {
                        return;
                    }
                    var cell = calendar.getCellBelowPoint(coords.x, coords.y);
                    calendar.clearShadow();
                    var resizing = DayPilotMonth.resizingEvent;
                    var width = void 0, start = void 0;
                    if (resizing.direction === 'w-resize') {
                        start = cell;
                        var endDate = resizing.event.end();
                        if (endDate.getDatePart() === endDate) {
                            endDate = endDate.addDays(-1);
                        }
                        var end = calendar.getCellFromDate(endDate);
                        width = calendar.getWidth(cell, end);
                    }
                    else {
                        start = calendar.getCellFromDate(resizing.event.start());
                        width = calendar.getWidth(start, cell);
                    }
                    if (width < 1) {
                        width = 1;
                    }
                    calendar.drawShadow(start.x, start.y, 0, width);
                }
                else if (DayPilotMonth.movingEvent) {
                    var coords = DayPilot.mo3(calendar.nav.top, ev);
                    if (!coords) {
                        return;
                    }
                    if (coords.x === DayPilotMonth.movingEvent.coords.x && coords.y === DayPilotMonth.movingEvent.coords.y) {
                        return;
                    }
                    var minDistance = 3;
                    var distance = Math.abs(coords.x - DayPilotMonth.movingEvent.coords.x) + Math.abs(coords.y - DayPilotMonth.movingEvent.coords.y);
                    if (distance <= minDistance) {
                        return;
                    }
                    var cell = calendar.getCellBelowPoint(coords.x, coords.y);
                    calendar.clearShadow();
                    var event_1 = DayPilotMonth.movingEvent.event;
                    var offset = DayPilotMonth.movingEvent.offset;
                    var width = calendar.cellMode ? 1 : DayPilot.DateUtil.daysSpan(event_1.start(), event_1.end()) + 1;
                    if (width < 1) {
                        width = 1;
                    }
                    calendar.drawShadow(cell.x, cell.y, 0, width, offset, event_1);
                }
                else if (DayPilotMonth.timeRangeSelecting) {
                    var coords = DayPilot.mo3(calendar.nav.top, ev);
                    if (!coords) {
                        return;
                    }
                    var cell = calendar.getCellBelowPoint(coords.x, coords.y);
                    calendar.clearShadow();
                    var start = DayPilotMonth.timeRangeSelecting;
                    var startIndex = start.y * 7 + start.x;
                    var cellIndex = cell.y * 7 + cell.x;
                    var width = Math.abs(cellIndex - startIndex) + 1;
                    if (width < 1) {
                        width = 1;
                    }
                    var shadowStart = startIndex < cellIndex ? start : cell;
                    DayPilotMonth.timeRangeSelecting.from = { x: shadowStart.x, y: shadowStart.y };
                    DayPilotMonth.timeRangeSelecting.width = width;
                    DayPilotMonth.timeRangeSelecting.moved = true;
                    calendar.drawShadow(shadowStart.x, shadowStart.y, 0, width, 0, null);
                }
            };
        };
        this._updateHeight = function () {
            this.nav.top.style.height = this.getHeight() + 'px';
            for (var x = 0; x < this.cells.length; x++) {
                for (var y = 0; y < this.cells[x].length; y++) {
                    this.cells[x][y].style.top = this.getRowTop(y) + 'px';
                    this.cells[x][y].style.height = this.rows[y].getHeight() + 'px';
                }
            }
        };
        this.getCellBelowPoint = function (x, y) {
            var columnWidth = Math.floor(this.nav.top.clientWidth / this.getColCount());
            var column = Math.min(Math.floor(x / columnWidth), this.getColCount() - 1);
            var row = null;
            var height = this.headerHeight;
            var relativeY = 0;
            for (var i = 0; i < this.rows.length; i++) {
                var baseHeight = height;
                height += this.rows[i].getHeight();
                if (y < height) {
                    relativeY = y - baseHeight;
                    row = i;
                    break;
                }
            }
            if (row === null) {
                row = this.rows.length - 1;
            }
            var cell = {};
            cell.x = column;
            cell.y = row;
            cell.relativeY = relativeY;
            return cell;
        };
        this.getCellFromDate = function (date) {
            var width = DayPilot.DateUtil.daysDiff(this.firstDate, date);
            var cell = { x: 0, y: 0 };
            while (width >= 7) {
                cell.y++;
                width -= 7;
            }
            cell.x = width;
            return cell;
        };
        this._drawTable = function () {
            var table = document.createElement("div");
            table.oncontextmenu = function () { return false; };
            this.nav.cells.appendChild(table);
            this.cells = [];
            for (var x = 0; x < this.getColCount(); x++) {
                this.cells[x] = [];
                var header = document.createElement("div");
                header.setAttribute("unselectable", "on");
                header.style.position = 'absolute';
                header.style.left = (this.getCellWidth() * x) + '%';
                header.style.width = (this.getCellWidth()) + '%';
                header.style.top = '0px';
                header.style.height = (this.headerHeight) + 'px';
                var dayIndex = x + this.getWeekStart();
                if (dayIndex > 6) {
                    dayIndex -= 7;
                }
                header.className = this._prefixCssClass("_header");
                var inner = document.createElement("div");
                inner.setAttribute("unselectable", "on");
                inner.innerHTML = resolved.locale().dayNames[dayIndex];
                header.appendChild(inner);
                inner.style.position = "absolute";
                inner.style.top = "0px";
                inner.style.bottom = "0px";
                inner.style.left = "0px";
                inner.style.right = "0px";
                inner.className = this._prefixCssClass("_header_inner");
                inner.innerHTML = resolved.locale().dayNames[dayIndex];
                table.appendChild(header);
                for (var y = 0; y < this.rows.length; y++) {
                    this._drawCell(x, y, table);
                }
            }
        };
        this._clearTable = function () {
            for (var x = 0; x < this.cells.length; x++) {
                for (var y = 0; y < this.cells[x].length; y++) {
                    var div = calendar.cells[x][y];
                    calendar._deleteCell(div);
                }
            }
            this.nav.cells.innerHTML = '';
        };
        this._deleteCell = function (div) {
            (function domRemove() {
                var domArgs = div.domArgs;
                div.domArgs = null;
                if (domArgs && typeof calendar.onBeforeCellDomRemove === "function") {
                    calendar.onBeforeCellDomRemove(domArgs);
                }
                if (domArgs && typeof calendar.onBeforeCellDomAdd === "function") {
                    var target = domArgs && domArgs._targetElement;
                    if (target) {
                        var isVue = calendar._vue._vueImport && isVueVNode(domArgs.element);
                        if (isVue) {
                            calendar._vue._renderingEvent = true;
                            calendar._vue._unmountVueComponent(target);
                            calendar._vue._renderingEvent = false;
                        }
                    }
                }
            })();
            div.onclick = null;
        };
        this._api2 = function () {
            return calendar.api === 2;
        };
        this._drawCell = function (x, y, table) {
            var row = this.rows[y];
            var d = this.firstDate.addDays(y * 7 + x);
            var date = d.getDay();
            var headerHtml = null;
            if (date === 1) {
                headerHtml = resolved.locale().monthNames[d.getMonth()] + ' ' + date;
            }
            else {
                headerHtml = "" + date;
            }
            var business = !calendar.isWeekend(d);
            var cellInfo = {
                "start": d,
                "end": d.addDays(1),
                "properties": {
                    "headerHtml": headerHtml,
                    "backColor": null,
                    "business": business,
                    "html": null
                }
            };
            var args = {};
            args.control = calendar;
            args.cell = cellInfo;
            if (typeof calendar.onBeforeCellRender === "function") {
                calendar.onBeforeCellRender(args);
            }
            var props = args.cell.properties;
            var cell = document.createElement("div");
            cell.setAttribute("unselectable", "on");
            cell.style.position = 'absolute';
            cell.style.cursor = 'default';
            cell.style.left = (this.getCellWidth() * x) + '%';
            cell.style.width = (this.getCellWidth()) + '%';
            cell.style.top = (this.getRowTop(y)) + 'px';
            cell.style.height = (row.getHeight()) + 'px';
            cell.className = this._prefixCssClass("_cell");
            if (props.business) {
                var businessCss = this._prefixCssClass("_cell_business");
                DayPilot.Util.addClass(cell, businessCss);
            }
            var inner = document.createElement("div");
            inner.setAttribute("unselectable", "on");
            cell.appendChild(inner);
            inner.style.position = "absolute";
            inner.style.left = "0px";
            inner.style.right = "0px";
            inner.style.top = "0px";
            inner.style.bottom = "0px";
            inner.className = this._prefixCssClass("_cell_inner");
            if (props.backColor) {
                inner.style.backgroundColor = args.cell.properties.backColor;
            }
            cell.onmousedown = function () {
                if (calendar.timeRangeSelectedHandling !== 'Disabled') {
                    calendar.clearShadow();
                    DayPilotMonth.timeRangeSelecting = { "root": calendar, "x": x, "y": y, "from": { x: x, y: y }, "width": 1 };
                }
            };
            cell.onclick = function () {
                var single = function (d) {
                    var start = new DayPilot.Date(d);
                    var end = start.addDays(1);
                    calendar._timeRangeSelectedDispatch(start, end);
                };
                if (calendar.timeRangeSelectedHandling !== 'Disabled') {
                    single(d);
                    return;
                }
            };
            var day = document.createElement("div");
            day.setAttribute("unselectable", "on");
            day.style.height = this.cellHeaderHeight + "px";
            day.className = this._prefixCssClass("_cell_header");
            day.onclick = function (ev) {
                if (calendar.cellHeaderClickHandling !== "Enabled") {
                    return;
                }
                ev.stopPropagation();
                var args = {};
                args.control = calendar;
                args.start = d;
                args.end = d.addDays(1);
                args.preventDefault = function () {
                    this.preventDefault.value = true;
                };
                if (typeof calendar.onCellHeaderClick === "function") {
                    calendar.onCellHeaderClick(args);
                    if (args.preventDefault.value) {
                        return;
                    }
                }
                if (typeof calendar.onCellHeaderClicked === "function") {
                    calendar.onCellHeaderClicked(args);
                }
            };
            day.innerHTML = props.headerHtml;
            inner.appendChild(day);
            if (props.html) {
                var html = document.createElement("div");
                html.style.height = (row.getHeight() - this.cellHeaderHeight) + 'px';
                html.style.overflow = 'hidden';
                html.innerHTML = props.html;
                inner.appendChild(html);
            }
            (function domAdd() {
                if (typeof calendar.onBeforeCellDomAdd !== "function" && typeof calendar.onBeforeCellDomRemove !== "function") {
                    return;
                }
                var args = {};
                args.control = calendar;
                args.cell = cellInfo;
                args.element = null;
                cell.domArgs = args;
                if (typeof calendar.onBeforeCellDomAdd === "function") {
                    calendar.onBeforeCellDomAdd(args);
                }
                if (args.element) {
                    var target = inner;
                    if (target) {
                        args._targetElement = target;
                        var isVueNode = isVueVNode(args.element);
                        if (isVueNode) {
                            if (!calendar._vue._vueImport) {
                                throw new DayPilot.Exception("Can't reach Vue");
                            }
                            calendar._vue._renderingEvent = true;
                            calendar._vue._renderVueNode(args.element, target);
                            calendar._vue._renderingEvent = false;
                        }
                        else {
                            target.appendChild(args.element);
                        }
                    }
                }
            })();
            this.cells[x][y] = cell;
            table.appendChild(cell);
        };
        this.getWeekStart = function () {
            if (calendar.weekStarts === 'Auto') {
                var locale = resolved.locale();
                if (locale) {
                    return locale.weekStarts;
                }
                else {
                    return 0;
                }
            }
            else {
                return calendar.weekStarts || 0;
            }
        };
        this.getColCount = function () {
            return 7;
        };
        this.getCellWidth = function () {
            return 14.285;
        };
        this.getRowTop = function (index) {
            var top = this.headerHeight;
            for (var i = 0; i < index; i++) {
                top += this.rows[i].getHeight();
            }
            return top;
        };
        this._callBack2 = function (action, data, parameters) {
            var envelope = {};
            envelope.action = action;
            envelope.parameters = parameters;
            envelope.data = data;
            envelope.header = this._getCallBackHeader();
            var commandstring = "JSON" + JSON.stringify(envelope);
            if (this.backendUrl) {
                DayPilot.request(this.backendUrl, this._callBackResponse, commandstring, this.ajaxError);
            }
        };
        this._callBackResponse = function (response) {
            calendar._updateView(response.responseText);
        };
        this._getCallBackHeader = function () {
            var h = {};
            h.control = "dpm";
            h.id = this.id;
            h.v = this.v;
            h.visibleStart = new DayPilot.Date(this.firstDate);
            h.visibleEnd = h.visibleStart.addDays(this.days);
            h.startDate = calendar.startDate;
            h.timeFormat = this.timeFormat;
            h.weekStarts = this.weekStarts;
            return h;
        };
        this.eventClickCallBack = function (e, data) {
            this._callBack2('EventClick', data, e);
        };
        this._eventClickDispatch = function (e) {
            DayPilotMonth.movingEvent = null;
            DayPilotMonth.resizingEvent = null;
            var div = e.currentTarget;
            e.cancelBubble = true;
            if (e.stopPropagation) {
                e.stopPropagation();
            }
            calendar.eventClickSingle(div, e);
        };
        this.eventClickSingle = function (div, ev) {
            var e = div.event;
            if (!e) {
                return;
            }
            if (!e.client.clickEnabled()) {
                return;
            }
            if (calendar._api2()) {
                var args_1 = {};
                args_1.e = e;
                args_1.control = calendar;
                args_1.div = div;
                args_1.originalEvent = ev;
                args_1.meta = ev.metaKey;
                args_1.ctrl = ev.ctrlKey;
                args_1.preventDefault = function () {
                    this.preventDefault.value = true;
                };
                if (typeof calendar.onEventClick === 'function') {
                    calendar._angular.apply(function () {
                        calendar.onEventClick(args_1);
                    });
                    if (args_1.preventDefault.value) {
                        return;
                    }
                }
                switch (calendar.eventClickHandling) {
                    case 'CallBack':
                        calendar.eventClickCallBack(e);
                        break;
                    case 'ContextMenu': {
                        var menu = e.client.contextMenu();
                        if (menu) {
                            menu.show(e);
                        }
                        else {
                            if (calendar.contextMenu) {
                                calendar.contextMenu.show(e);
                            }
                        }
                        break;
                    }
                }
                if (typeof calendar.onEventClicked === 'function') {
                    calendar._angular.apply(function () {
                        calendar.onEventClicked(args_1);
                    });
                }
            }
            else {
                switch (calendar.eventClickHandling) {
                    case 'CallBack':
                        calendar.eventClickCallBack(e);
                        break;
                    case 'JavaScript':
                        calendar.onEventClick(e);
                        break;
                }
            }
        };
        this._onEventContextMenu = function (ev) {
            var e = ev.currentTarget;
            calendar._eventRightClickDispatch(e.event);
            return false;
        };
        this._eventRightClickDispatch = function (e) {
            this.event = e;
            if (!e.client.rightClickEnabled()) {
                return false;
            }
            var args = {};
            args.e = e;
            args.preventDefault = function () {
                this.preventDefault.value = true;
            };
            if (typeof calendar.onEventRightClick === 'function') {
                calendar.onEventRightClick(args);
                if (args.preventDefault.value) {
                    return;
                }
            }
            switch (calendar.eventRightClickHandling) {
                case 'ContextMenu': {
                    var menu = e.client.contextMenu();
                    if (menu) {
                        menu.show(e);
                    }
                    else {
                        if (calendar.contextMenu) {
                            calendar.contextMenu.show(this.event);
                        }
                    }
                    break;
                }
            }
            if (typeof calendar.onEventRightClicked === 'function') {
                calendar.onEventRightClicked(args);
            }
            return false;
        };
        this._eventDeleteDispatch = function (e) {
            if (calendar._api2()) {
                var args_2 = {};
                args_2.e = e;
                args_2.control = calendar;
                args_2.preventDefault = function () {
                    this.preventDefault.value = true;
                };
                if (typeof calendar.onEventDelete === 'function') {
                    calendar._angular.apply(function () {
                        calendar.onEventDelete(args_2);
                    });
                    if (args_2.preventDefault.value) {
                        return;
                    }
                }
                switch (calendar.eventDeleteHandling) {
                    case 'CallBack':
                        calendar.eventDeleteCallBack(e);
                        break;
                    case 'PostBack':
                        calendar.eventDeletePostBack(e);
                        break;
                    case 'Update':
                        calendar.events.remove(e);
                        break;
                }
                if (typeof calendar.onEventDeleted === 'function') {
                    calendar._angular.apply(function () {
                        calendar.onEventDeleted(args_2);
                    });
                }
            }
            else {
                switch (calendar.eventDeleteHandling) {
                    case 'PostBack':
                        calendar.eventDeletePostBack(e);
                        break;
                    case 'CallBack':
                        calendar.eventDeleteCallBack(e);
                        break;
                    case 'JavaScript':
                        calendar.onEventDelete(e);
                        break;
                }
            }
        };
        this.eventDeleteCallBack = function (e, data) {
            this._callBack2('EventDelete', data, e);
        };
        this.eventDeletePostBack = function (e, data) {
            this._postBack2('EventDelete', data, e);
        };
        this.eventMoveCallBack = function (e, newStart, newEnd, data, position) {
            if (!newStart) {
                throw 'newStart is null';
            }
            if (!newEnd) {
                throw 'newEnd is null';
            }
            var params = {};
            params.e = e;
            params.newStart = newStart;
            params.newEnd = newEnd;
            params.position = position;
            this._callBack2('EventMove', data, params);
        };
        this._eventMoveDispatch = function (e, x, y, offset, ev) {
            var startOffset = e.start().getTimePart();
            var endDate = e.end().getDatePart();
            if (endDate.getTime() !== e.end().getTime()) {
                endDate = endDate.addDays(1);
            }
            var endOffset = DayPilot.DateUtil.diff(e.end(), endDate);
            var boxStart = this.getDateFromCell(x, y);
            boxStart = boxStart.addDays(-offset);
            var width = DayPilot.DateUtil.daysSpan(e.start(), e.end()) + 1;
            var boxEnd = boxStart.addDays(width);
            var newStart = boxStart.addTime(startOffset);
            var newEnd = boxEnd.addTime(endOffset);
            if (calendar._api2()) {
                var args_3 = {};
                args_3.e = e;
                args_3.control = calendar;
                args_3.newStart = newStart;
                args_3.newEnd = newEnd;
                args_3.ctrl = ev.ctrlKey;
                args_3.shift = ev.shiftKey;
                args_3.preventDefault = function () {
                    this.preventDefault.value = true;
                };
                if (typeof calendar.onEventMove === 'function') {
                    calendar._angular.apply(function () {
                        calendar.onEventMove(args_3);
                    });
                    if (args_3.preventDefault.value) {
                        return;
                    }
                }
                switch (calendar.eventMoveHandling) {
                    case 'CallBack':
                        calendar.eventMoveCallBack(e, newStart, newEnd);
                        break;
                    case 'Update':
                        e.start(newStart);
                        e.end(newEnd);
                        calendar.events.update(e);
                        break;
                }
                if (typeof calendar.onEventMoved === 'function') {
                    calendar._angular.apply(function () {
                        calendar.onEventMoved(args_3);
                    });
                }
            }
            else {
                switch (calendar.eventMoveHandling) {
                    case 'CallBack':
                        calendar.eventMoveCallBack(e, newStart, newEnd);
                        break;
                    case 'JavaScript':
                        calendar.onEventMove(e, newStart, newEnd);
                        break;
                }
            }
        };
        this.eventResizeCallBack = function (e, newStart, newEnd, data) {
            if (!newStart) {
                throw 'newStart is null';
            }
            if (!newEnd) {
                throw 'newEnd is null';
            }
            var params = {};
            params.e = e;
            params.newStart = newStart;
            params.newEnd = newEnd;
            this._callBack2('EventResize', data, params);
        };
        this._eventResizeDispatch = function (e, start, width) {
            var startOffset = e.start().getTimePart();
            var endDate = e.end().getDatePart();
            if (endDate.getTime() !== e.end().getTime()) {
                endDate = endDate.addDays(1);
            }
            var endOffset = DayPilot.DateUtil.diff(e.end(), endDate);
            var boxStart = this.getDateFromCell(start.x, start.y);
            var boxEnd = boxStart.addDays(width);
            var newStart = boxStart.addTime(startOffset);
            var newEnd = boxEnd.addTime(endOffset);
            if (calendar._api2()) {
                var args_4 = {};
                args_4.e = e;
                args_4.control = calendar;
                args_4.newStart = newStart;
                args_4.newEnd = newEnd;
                args_4.preventDefault = function () {
                    this.preventDefault.value = true;
                };
                if (typeof calendar.onEventResize === 'function') {
                    calendar._angular.apply(function () {
                        calendar.onEventResize(args_4);
                    });
                    if (args_4.preventDefault.value) {
                        return;
                    }
                }
                switch (calendar.eventResizeHandling) {
                    case 'CallBack':
                        calendar.eventResizeCallBack(e, newStart, newEnd);
                        break;
                    case 'Update':
                        e.start(newStart);
                        e.end(newEnd);
                        calendar.events.update(e);
                        break;
                }
                if (typeof calendar.onEventResized === 'function') {
                    calendar._angular.apply(function () {
                        calendar.onEventResized(args_4);
                    });
                }
            }
            else {
                switch (calendar.eventResizeHandling) {
                    case 'CallBack':
                        calendar.eventResizeCallBack(e, newStart, newEnd);
                        break;
                    case 'JavaScript':
                        calendar.onEventResize(e, newStart, newEnd);
                        break;
                }
            }
        };
        this.timeRangeSelectedCallBack = function (start, end, data) {
            var range = {};
            range.start = start;
            range.end = end;
            this._callBack2('TimeRangeSelected', data, range);
        };
        this._timeRangeSelectedDispatch = function (start, end) {
            if (this._api2()) {
                var args_5 = {};
                args_5.control = calendar;
                args_5.start = start;
                args_5.end = end;
                args_5.preventDefault = function () {
                    this.preventDefault.value = true;
                };
                if (typeof calendar.onTimeRangeSelect === 'function') {
                    calendar._angular.apply(function () {
                        calendar.onTimeRangeSelect(args_5);
                    });
                    if (args_5.preventDefault.value) {
                        return;
                    }
                }
                switch (calendar.timeRangeSelectedHandling) {
                    case 'CallBack':
                        calendar.timeRangeSelectedCallBack(start, end);
                        break;
                }
                if (typeof calendar.onTimeRangeSelected === 'function') {
                    calendar._angular.apply(function () {
                        calendar.onTimeRangeSelected(args_5);
                    });
                }
            }
            else {
                switch (calendar.timeRangeSelectedHandling) {
                    case 'CallBack':
                        calendar.timeRangeSelectedCallBack(start, end);
                        break;
                    case 'JavaScript':
                        calendar.onTimeRangeSelected(start, end);
                        break;
                }
            }
        };
        this._angular = {};
        this._angular.scope = null;
        this._angular.notify = function () {
            if (calendar._angular.scope) {
                calendar._angular.scope["$apply"]();
            }
        };
        this._angular.apply = function (f) {
            f();
        };
        this.clearSelection = function () {
            calendar.clearShadow();
        };
        this.commandCallBack = function (command, data) {
            var params = {};
            params.command = command;
            this._callBack2('Command', data, params);
        };
        this.isWeekend = function (date) {
            date = new DayPilot.Date(date);
            var sunday = 0;
            var saturday = 6;
            if (date.dayOfWeek() === sunday) {
                return true;
            }
            if (date.dayOfWeek() === saturday) {
                return true;
            }
            return false;
        };
        this._resolved.locale = function () {
            var found = DayPilot.Locale.find(calendar.locale);
            if (!found) {
                return DayPilot.Locale.US;
            }
            return found;
        };
        this._resolved._xssProtectionEnabled = function () {
            return calendar.xssProtection !== "Disabled";
        };
        this.debug = function (msg) {
            if (!this.debuggingEnabled) {
                return;
            }
            if (!calendar.debugMessages) {
                calendar.debugMessages = [];
            }
            calendar.debugMessages.push(msg);
            if (typeof console !== 'undefined') {
                console.log(msg);
            }
        };
        this.dispose = function () {
            var c = calendar;
            if (c._disposed) {
                return;
            }
            c._disposed = true;
            c._clearTable();
            c._deleteEvents();
            c.nav.top.removeAttribute("style");
            c.nav.top.removeAttribute("class");
            c.nav.top.innerHTML = '';
            c.nav.top.dp = null;
            c.nav.top.onmousemove = null;
            c.nav.top = null;
        };
        this.disposed = function () {
            return this._disposed;
        };
        this._registerGlobalHandlers = function () {
            if (!DayPilotMonth.globalHandlers) {
                DayPilotMonth.globalHandlers = true;
                DayPilot.re(document, 'mouseup', DayPilotMonth.gMouseUp);
            }
        };
        this.loadFromServer = function () {
            if (this.backendUrl || typeof WebForm_DoCallback === 'function') {
                return (typeof calendar.events.list === 'undefined') || (!calendar.events.list);
            }
            else {
                return false;
            }
        };
        this._show = function () {
            if (this.nav.top.style.visibility === 'hidden') {
                this.nav.top.style.visibility = 'visible';
            }
        };
        this.show = function () {
            calendar.visible = true;
            calendar.nav.top.style.display = '';
        };
        this.hide = function () {
            calendar.visible = false;
            calendar.nav.top.style.display = 'none';
        };
        this._loadTop = function () {
            if (this.id && this.id.tagName) {
                this.nav.top = this.id;
            }
            else if (typeof this.id === "string") {
                this.nav.top = document.getElementById(this.id);
                if (!this.nav.top) {
                    throw "DayPilot.Month: The placeholder element not found: '" + this.id + "'.";
                }
            }
            else {
                throw "DayPilot.Month() constructor requires the target element or its ID as a parameter";
            }
        };
        this._initShort = function () {
            this._prepareRows();
            this._drawTop();
            this._drawTable();
            this._registerGlobalHandlers();
            this._callBack2('Init');
        };
        this._xssTextHtml = function (text, html) {
            if (calendar._resolved._xssProtectionEnabled()) {
                return DayPilot.Util.escapeTextHtml(text, html);
            }
            if (!DayPilot.Util.isNullOrUndefined(html)) {
                return html;
            }
            if (DayPilot.Util.isNullOrUndefined(text)) {
                return "";
            }
            return text;
        };
        this._vue = {};
        this._vue._vueImport = null;
        this._vue._renderVueComponent = function (component, target, props) {
            var vue = calendar._vue._vueImport;
            if (typeof vue.createVNode === "function" && typeof vue.render === "function") {
                var vnode = vue.createVNode(component, props);
                vue.render(vnode, target);
            }
        };
        this._vue._renderVueNode = function (vnode, target, props) {
            var vue = calendar._vue._vueImport;
            if (typeof vue.render === "function") {
                var toRender = vnode;
                if (DayPilot.isArray(vnode)) {
                    toRender = vue.h("div", props, vnode);
                }
                vue.render(toRender, target);
            }
        };
        this._vue._unmountVueComponent = function (target) {
            var vue = calendar._vue._vueImport;
            if (typeof vue.render === "function") {
                vue.render(null, target);
            }
        };
        this.internal = {};
        this.internal.loadOptions = this._loadOptions;
        this.internal.xssTextHtml = calendar._xssTextHtml;
        this.internal.upd = function (options) {
            calendar.update(options);
        };
        this.internal.enableVue = function (vue) {
            calendar._vue._vueImport = vue;
        };
        this.internal.vueRef = function () {
            return calendar._vue._vueImport;
        };
        this.internal.vueRendering = function () {
            return calendar._vue._renderingEvent;
        };
        this.init = function () {
            this._loadTop();
            var loadFromServer = this.loadFromServer();
            if (loadFromServer) {
                this._initShort();
                return;
            }
            this._prepareRows();
            this._loadEvents();
            this._drawTop();
            this._drawTable();
            this._show();
            this._drawEvents();
            this._registerGlobalHandlers();
            this.fireAfterRenderDetached(null, false);
            this._initialized = true;
            return this;
        };
        this.Init = this.init;
        Object.defineProperty(this, 'durationBarVisible', { get: function () { return calendar.eventBarVisible; } });
        this._loadOptions(options);
    };
    DayPilotMonth.gMouseUp = function (ev) {
        if (DayPilotMonth.movingEvent) {
            var src = DayPilotMonth.movingEvent;
            if (!src.event) {
                return;
            }
            if (!src.event.calendar) {
                return;
            }
            if (!src.event.calendar.shadow) {
                return;
            }
            if (!src.event.calendar.shadow.start) {
                return;
            }
            var calendar = DayPilotMonth.movingEvent.event.calendar;
            var e = DayPilotMonth.movingEvent.event;
            var start = calendar.shadow.start;
            var position = calendar.shadow.position;
            var offset = DayPilotMonth.movingEvent.offset;
            calendar.clearShadow();
            DayPilotMonth.movingEvent = null;
            calendar._eventMoveDispatch(e, start.x, start.y, offset, ev, position);
            ev.cancelBubble = true;
            if (ev.stopPropagation) {
                ev.stopPropagation();
            }
            DayPilotMonth.movingEvent = null;
            return false;
        }
        else if (DayPilotMonth.resizingEvent) {
            var src = DayPilotMonth.resizingEvent;
            if (!src.event) {
                return;
            }
            if (!src.event.calendar) {
                return;
            }
            if (!src.event.calendar.shadow) {
                return;
            }
            if (!src.event.calendar.shadow.start) {
                return;
            }
            var calendar = DayPilotMonth.resizingEvent.event.calendar;
            var e = DayPilotMonth.resizingEvent.event;
            var start = calendar.shadow.start;
            var width = calendar.shadow.width;
            calendar.clearShadow();
            DayPilotMonth.resizingEvent = null;
            calendar._eventResizeDispatch(e, start, width);
            ev.cancelBubble = true;
            DayPilotMonth.resizingEvent = null;
            return false;
        }
        else if (DayPilotMonth.timeRangeSelecting) {
            if (DayPilotMonth.timeRangeSelecting.moved) {
                var sel = DayPilotMonth.timeRangeSelecting;
                var calendar = sel.root;
                var start = new DayPilot.Date(calendar.getDateFromCell(sel.from.x, sel.from.y));
                var end = start.addDays(sel.width);
                calendar._timeRangeSelectedDispatch(start, end);
            }
            DayPilotMonth.timeRangeSelecting = null;
        }
    };
    DayPilot.Month = DayPilotMonth.Month;
    if (typeof jQuery !== 'undefined') {
        (function ($) {
            $.fn.daypilotMonth = function (options) {
                var first = null;
                var j = this.each(function () {
                    if (this.daypilot) {
                        return;
                    }
                    ;
                    var daypilot = new DayPilot.Month(this.id);
                    this.daypilot = daypilot;
                    for (var name_4 in options) {
                        daypilot[name_4] = options[name_4];
                    }
                    daypilot.Init();
                    if (!first) {
                        first = daypilot;
                    }
                });
                if (this.length === 1) {
                    return first;
                }
                else {
                    return j;
                }
            };
        })(jQuery);
    }
    (function registerAngularModule() {
        var app = DayPilot.am();
        if (!app) {
            return;
        }
        app.directive("daypilotMonth", ['$parse', function ($parse) {
                return {
                    "restrict": "E",
                    "template": "<div></div>",
                    "replace": true,
                    "link": function (scope, element, attrs) {
                        var calendar = new DayPilot.Month(element[0]);
                        calendar._angular.scope = scope;
                        calendar.init();
                        var oattr = attrs["id"];
                        if (oattr) {
                            scope[oattr] = calendar;
                        }
                        var pas = attrs["publishAs"];
                        if (pas) {
                            var getter = $parse(pas);
                            var setter = getter.assign;
                            setter(scope, calendar);
                        }
                        for (var name_5 in attrs) {
                            if (name_5.indexOf("on") === 0) {
                                (function (name) {
                                    calendar[name] = function (args) {
                                        var f = $parse(attrs[name]);
                                        scope["$apply"](function () {
                                            f(scope, { "args": args });
                                        });
                                    };
                                })(name_5);
                            }
                        }
                        var watch = scope["$watch"];
                        var config = attrs["config"] || attrs["daypilotConfig"];
                        var events = attrs["events"] || attrs["daypilotEvents"];
                        watch.call(scope, config, function (value) {
                            for (var name_6 in value) {
                                calendar[name_6] = value[name_6];
                            }
                            calendar.update();
                        }, true);
                        watch.call(scope, events, function (value) {
                            calendar.events.list = value;
                            calendar.update();
                        }, true);
                    }
                };
            }]);
    })();
    if (typeof Sys !== 'undefined' && Sys.Application && Sys.Application.notifyScriptLoaded) {
        Sys.Application.notifyScriptLoaded();
    }
})(DayPilot);
